use strict;
use warnings;
use lib 'lib';
use Test::More;

use JSON::WebToken;
use Crypt::CBC;
use Crypt::OpenSSL::AES;
use Crypt::OpenSSL::RSA;
use Crypt::OpenSSL::Bignum;
use Digest::SHA qw(hmac_sha256);

my $input = join q{} => (
    'eyJhbGciOiJSU0ExXzUiLCJlbmMiOiJBMTI4Q0JDIiwiaW50IjoiSFMyNTYiLCJpdiI6IkF4W',
    'ThEQ3REYUdsc2JHbGpiM1JvWlEifQ',
    '.',
    'VjBkk22MjrFUMUl8ItbS8CjKjku4HQz4RiHD0eVG4dir-7XbDkPr1q6YtnN1X-av1EKmEnsrb',
    'hSxTvqtY4oEbWKLoEQ7zVm_0BW-rnwxdwrj4QJrhXGnqIL6bC4waZVJqYhVQIahVWSQsCRcS1',
    'oYXA-2GhT6rk91y118DUkhGDsvdK2_hQsNGE6BQVN1i-XwUoz5sM6_0PRQ1FsYnJATMjVZfa4',
    'otHiooZ_KcOlSWIDxhMDqfPOu60--1ej0eZByO7Ar_IZvzPAWqJ9agGFQIVGRZviXhN0WeErq',
    '9fVTcgeSUPsmurRSTYhTrNFLojqPqqk8pI61kn8GmZxA80-RUQ',
    '.',
    '7kLQQst655TUxmDzysjRLXnD-nfLK5EQK7ODAUkwxc0aRb9NOgu0EMJgOR6Vz8eNbaf8six_O',
    'P6BRyUTYrCkH73-inD6Rc-7vc9eC5fcfSM',
    '.',
    'COyXNSm-CdfAL22WIKcoyCgQwb85aLW3ttDkzNj_1Wg',
);

my $plaintext = pack 'C*' => @{ [
    78,  111, 119, 32,  105, 115, 32,  116, 104, 101, 32,  116, 105, 109,
    101, 32,  102, 111, 114, 32,  97,  108, 108, 32,  103, 111, 111, 100,
    32,  109, 101, 110, 32,  116, 111, 32,  99,  111, 109, 101, 32,  116,
    111, 32,  116, 104, 101, 32,  97,  105, 100, 32,  111, 102, 32,  116,
    104, 101, 105, 114, 32,  99,  111, 117, 110, 116, 114, 121, 46
] };

my $iv = pack 'C*' => @{ [
    3, 22, 60, 12, 43, 67, 104, 105, 108, 108, 105, 99, 111, 116, 104, 101
] };

my $cmk = pack 'C*' => @{ [
    4,   211, 31,  197, 84,  157, 252, 254, 11,  100, 157, 250,
    63,  170, 106, 206, 107, 124, 212, 45,  111, 107, 9,   219,
    200, 177, 0,   240, 143, 156, 44,  207
] };

my $rsa = do {
    my $n = pack 'C*' => @{ [
        177, 119, 33, 13, 164, 30, 108, 121, 207, 136, 107,
        242, 12,  224, 19,  226, 198, 134, 17,  71,  173, 75,  42,
        61,  48,  162, 206, 161, 97,  108, 185, 234, 226, 219,
        118, 206, 118, 5,   169, 224, 60,  181, 90,  85,  51,  123,
        6,   224, 4,   122, 29,  230, 151, 12,  244, 127, 121, 25,
        4,   85,  220, 144, 215, 110, 130, 17,  68,  228, 129,
        138, 7,   130, 231, 40,  212, 214, 17,  179, 28,  124,
        151, 178, 207, 20,  14,  154, 222, 113, 176, 24,  198,
        73,  211, 113, 9,   33,  178, 80,  13,  25,  21,  25,  153,
        212, 206, 67,  154, 147, 70,  194, 192, 183, 160, 83,
        98,  236, 175, 85,  23,  97,  75,  199, 177, 73,  145, 50,
        253, 206, 32,  179, 254, 236, 190, 82,  73,  67,  129,
        253, 252, 220, 108, 136, 138, 11,  192, 1,   36,  239,
        228, 55,  81,  113, 17,  25,  140, 63,  239, 146, 3,   172,
        96,  60,  227, 233, 64,  255, 224, 173, 225, 228, 229,
        92,  112, 72,  99,  97,  26,  87,  187, 123, 46,  50,  90,
        202, 117, 73,  10,  153, 47,  224, 178, 163, 77,  48,  46,
        154, 33,  148, 34,  228, 33,  172, 216, 89,  46,  225,
        127, 68,  146, 234, 30,  147, 54,  146, 5,   133, 45,  78,
        254, 85,  55,  75,  213, 86,  194, 218, 215, 163, 189,
        194, 54, 6,  83,  36, 18, 153, 53, 7,   48,  89, 35, 66,
        144, 7,  65, 154, 13, 97, 75,  55, 230, 132, 3,  13,
        239, 71
    ] };
    my $e = pack 'C*' => @{ [1, 0, 1] };
    my $d = pack 'C*' => @{ [
        84,  80, 150, 58,  165, 235, 242, 123, 217, 55, 38,
        154, 36, 181, 221, 156, 211, 215, 100, 164, 90, 88,
        40,  228, 83,  148, 54,  122, 4,   16,  165, 48,  76,  194,
        26,  107, 51,  53,  179, 165, 31,  18,  198, 173, 78,  61,
        56,  97,  252, 158, 140, 80,  63,  25,  223, 156, 36,  203,
        214, 252, 120, 67,  180, 167, 3,   82,  243, 25,  97,  214,
        83,  133, 69,  16,  104, 54,  160, 200, 41,  83,  164, 187,
        70,  153, 111, 234, 242, 158, 175, 28,  198, 48,  211,
        45,  148, 58,  23,  62,  227, 74,  52,  117, 42,  90,  41,
        249, 130, 154, 80,  119, 61,  26,  193, 40,  125, 10,
        152, 174, 227, 225, 205, 32,  62,  66,  6,   163, 100, 99,
        219, 19,  253, 25,  105, 80,  201, 29,  252, 157, 237,
        69,  1,   80,  171, 167, 20,  196, 156, 109, 249, 88,  0,
        3,   152, 38,  165, 72,  87,  6,   152, 71,  156, 214, 16,
        71,  30,  82,  51,  103, 76,  218, 63,  9,   84,  163, 249,
        91,  215, 44,  238, 85,  101, 240, 148, 1,   82,  224, 91,
        135, 105, 127, 84,  171, 181, 152, 210, 183, 126, 24,
        46,  196, 90,  173, 38,  245, 219, 186, 222, 27,  240,
        212, 194, 15,  66,  135, 226, 178, 190, 52,  245, 74,
        65,  224, 81,  100, 85,  25,  204, 165, 203, 187, 175,
        84,  100, 82,  15,  11,  23,  202, 151, 107, 54,  41,  207,
        3,   136, 229, 134, 131, 93,  139, 50,  182, 204, 93,
        130, 89
    ] };

    my $rsa = Crypt::OpenSSL::RSA->new_key_from_parameters(map {
        Crypt::OpenSSL::Bignum->new_from_bin($_);
    } $n, $e, $d);
    $rsa->use_pkcs1_padding;
    $rsa;
};

my $result = pack 'C*' => @{ [
    32,  242, 63,  207, 94,  246, 133, 37,  135, 48,  88,  4,   15,  193,
    6,   244, 51,  58,  132, 133, 212, 255, 163, 90,  59,  80,  200, 152,
    41,  244, 188, 215, 174, 160, 26,  188, 227, 180, 165, 234, 172, 63,
    24,  116, 152, 28,  149, 16,  94,  213, 201, 171, 180, 191, 11,  21,
    149, 172, 143, 54,  194, 58,  206, 201, 164, 28,  107, 155, 75,  101,
    22,  92,  227, 144, 95,  40,  119, 170, 7,   36,  225, 40,  141, 186,
    213, 7,   175, 16,  174, 122, 75,  32,  48,  193, 119, 202, 41,  152,
    210, 190, 68,  57,  119, 4,   197, 74,  7,   242, 239, 170, 204, 73,
    75,  213, 202, 113, 216, 18,  23,  66,  106, 208, 69,  244, 117, 147,
    2,   37,  207, 199, 184, 96,  102, 44,  70,  212, 87,  143, 253, 0,
    166, 59,  41,  115, 217, 80,  165, 87,  38,  5,   9,   184, 202, 68,
    67,  176, 4,   87,  254, 166, 227, 88,  124, 238, 249, 75,  114, 205,
    148, 149, 45,  78,  193, 134, 64,  189, 168, 76,  170, 76,  176, 72,
    148, 77,  215, 159, 146, 55,  189, 213, 85,  253, 135, 200, 59,  247,
    79,  37,  22,  200, 32,  110, 53,  123, 54,  39,  9,   178, 231, 238,
    95,  25,  211, 143, 87,  220, 88,  138, 209, 13,  227, 72,  58,  102,
    164, 136, 241, 14,  14,  45,  32,  77,  44,  244, 162, 239, 150, 248,
    181, 138, 251, 116, 245, 205, 137, 78,  34,  34,  10,  6,   59,  4,
    197, 2,   153, 251
] };

is $rsa->decrypt($rsa->encrypt($cmk)), $cmk;
is $rsa->decrypt($result), $cmk;

my $cek = pack 'C*' => @{ [
    249, 255, 87, 218, 224, 223, 221, 53, 204, 121, 166, 130, 195, 184,
    50, 69
] };

my $cik = pack 'C*' => @{ [
    218, 209, 130, 50,  169, 45,  70,  214, 29,  187, 123, 20,
    3,   158, 111, 122, 182, 94,  57,  133, 245, 76,  97,  44,
    193, 80,  81,  246, 115, 177, 225, 159
] };

my $ciphertext = pack 'C*' => @{ [
    253, 159, 221, 142, 82,  40,  11,  131, 3,   72,  34,  162, 173, 229,
    146, 217, 183, 173, 139, 132, 58,  137, 33,  182, 82,  49,  110, 141,
    11,  221, 207, 239, 207, 65,  213, 28,  20,  217, 14,  186, 87,  160,
    15,  160, 96,  142, 7,   69,  46,  55,  129, 224, 113, 206, 59,  181,
    7,   188, 255, 15,  16,  59,  180, 107, 75,  0,   217, 175, 254, 8,
    141, 48,  217, 132, 16,  217, 4,   30,  223, 147
] };

my $cbc = Crypt::CBC->new({
    literal_key => 1,
    header      => 'none',
    key         => $cek,
    keysize     => 128 / 8,
    iv          => $iv,
    cipher      => 'Crypt::OpenSSL::AES',
});
my $expects_plaintext = $cbc->decrypt($ciphertext);
is $plaintext, $expects_plaintext;

my $signing_body = pack 'C*' => @{ [
    101, 121, 74,  104, 98,  71,  99,  105, 79,  105, 74,  83,  85,  48,
    69,  120, 88,  122, 85,  105, 76,  67,  74,  108, 98,  109, 77,  105,
    79,  105, 74,  66,  77,  84,  73,  52,  81,  48,  74,  68,  73,  105,
    119, 105, 97,  87,  53,  48,  73,  106, 111, 105, 83,  70,  77,  121,
    78,  84,  89,  105, 76,  67,  74,  112, 100, 105, 73,  54,  73,  107,
    70,  52,  87,  84,  104, 69,  81,  51,  82,  69,  89,  85,  100, 115,
    99,  50,  74,  72,  98,  71,  112, 105, 77,  49,  74,  118, 87,  108,
    69,  105, 102, 81,  46,  73,  80,  73,  95,  122, 49,  55,  50,  104,
    83,  87,  72,  77,  70,  103, 69,  68,  56,  69,  71,  57,  68,  77,
    54,  104, 73,  88,  85,  95,  54,  78,  97,  79,  49,  68,  73,  109,
    67,  110, 48,  118, 78,  101, 117, 111, 66,  113, 56,  52,  55,  83,
    108, 54,  113, 119, 95,  71,  72,  83,  89,  72,  74,  85,  81,  88,
    116, 88,  74,  113, 55,  83,  95,  67,  120, 87,  86,  114, 73,  56,
    50,  119, 106, 114, 79,  121, 97,  81,  99,  97,  53,  116, 76,  90,
    82,  90,  99,  52,  53,  66,  102, 75,  72,  101, 113, 66,  121, 84,
    104, 75,  73,  50,  54,  49,  81,  101, 118, 69,  75,  53,  54,  83,
    121, 65,  119, 119, 88,  102, 75,  75,  90,  106, 83,  118, 107, 81,
    53,  100, 119, 84,  70,  83,  103, 102, 121, 55,  54,  114, 77,  83,
    85,  118, 86,  121, 110, 72,  89,  69,  104, 100, 67,  97,  116, 66,
    70,  57,  72,  87,  84,  65,  105, 88,  80,  120, 55,  104, 103, 90,
    105, 120, 71,  49,  70,  101, 80,  95,  81,  67,  109, 79,  121, 108,
    122, 50,  86,  67,  108, 86,  121, 89,  70,  67,  98,  106, 75,  82,
    69,  79,  119, 66,  70,  102, 45,  112, 117, 78,  89,  102, 79,  55,
    53,  83,  51,  76,  78,  108, 74,  85,  116, 84,  115, 71,  71,  81,
    76,  50,  111, 84,  75,  112, 77,  115, 69,  105, 85,  84,  100, 101,
    102, 107, 106, 101, 57,  49,  86,  88,  57,  104, 56,  103, 55,  57,
    48,  56,  108, 70,  115, 103, 103, 98,  106, 86,  55,  78,  105, 99,
    74,  115, 117, 102, 117, 88,  120, 110, 84,  106, 49,  102, 99,  87,
    73,  114, 82,  68,  101, 78,  73,  79,  109, 97,  107, 105, 80,  69,
    79,  68,  105, 48,  103, 84,  83,  122, 48,  111, 117, 45,  87,  45,
    76,  87,  75,  45,  51,  84,  49,  122, 89,  108, 79,  73,  105, 73,
    75,  66,  106, 115, 69,  120, 81,  75,  90,  45,  119, 46,  95,  90,
    95,  100, 106, 108, 73,  111, 67,  52,  77,  68,  83,  67,  75,  105,
    114, 101, 87,  83,  50,  98,  101, 116, 105, 52,  81,  54,  105, 83,
    71,  50,  85,  106, 70,  117, 106, 81,  118, 100, 122, 45,  95,  80,
    81,  100, 85,  99,  70,  78,  107, 79,  117, 108, 101, 103, 68,  54,
    66,  103, 106, 103, 100, 70,  76,  106, 101, 66,  52,  72,  72,  79,
    79,  55,  85,  72,  118, 80,  56,  80,  69,  68,  117, 48,  97,  48,
    115, 65,  50,  97,  95,  45,  67,  73,  48,  119, 50,  89,  81,  81,
    50,  81,  81,  101, 51,  53,  77
] };

my $integrity = pack 'C*' => @{ [
    115, 141, 100, 225, 62,  30,  2,  0,   130, 183, 173, 230,
    241, 147, 102, 136, 232, 167, 49, 200, 133, 23,  42,  78,
    22,  155, 226, 119, 184, 186, 15, 73
] };

my $expects_integrity = hmac_sha256 $signing_body, $cik;
is $integrity, $expects_integrity;


done_testing;

# IPI_z172hSWHMFgED8EG9DM6hIXU_6NaO1DImCn0vNeuoBq847Sl6qw_GHSYHJUQ
# XtXJq7S_CxWVrI82wjrOyaQca5tLZRZc45BfKHeqByThKI261QevEK56SyAwwXfK
# KZjSvkQ5dwTFSgfy76rMSUvVynHYEhdCatBF9HWTAiXPx7hgZixG1FeP_QCmOylz
# 2VClVyYFCbjKREOwBFf-puNYfO75S3LNlJUtTsGGQL2oTKpMsEiUTdefkje91VX9
# h8g7908lFsggbjV7NicJsufuXxnTj1fcWIrRDeNIOmakiPEODi0gTSz0ou-W-LWK
# -3T1zYlOIiIKBjsExQKZ-w
# 
