use strict;
use warnings;
use Test::More tests => 8;
use XML::Enc;
use MIME::Base64 qw/decode_base64/;

my $base64 = <<'SAML3DES_RSA-OAEP-MGF1P';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SAML3DES_RSA-OAEP-MGF1P

my $xml = decode_base64($base64);

ok($xml, "Got encrypted 3DES RSA-OAEP-MGF1P SAML2 Assertion");

my $decrypter = XML::Enc->new(
    {
        key                 => 't/sign-private.pem',
        no_xml_declaration  => 1
    }
);

ok($decrypter->decrypt($xml) =~ /Af49573f11706b4/, "Successfully Decrypted 3DES RSA-OAEP-MGF1P SAML2 Assertion");

$base64 = <<'SAMLAES128_RSA-OAEP-MGF1P';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SAMLAES128_RSA-OAEP-MGF1P

$xml = decode_base64($base64);

ok($xml, "Got encrypted AES128 RSA-OAEP-MGF1P SAML2 Assertion");

$decrypter = XML::Enc->new(
    {
        key                 => 't/sign-private.pem',
        no_xml_declaration  => 1
    }
);

ok($decrypter->decrypt($xml) =~ /Ac43ac806fc1e00b9f95/, "Successfully Decrypted AES128 RSA-OAEP-MGF1P Assertion");


$base64 = <<'SAMLAES196_RSA-OAEP-MGF1P';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SAMLAES196_RSA-OAEP-MGF1P

$xml = decode_base64($base64);

ok($xml, "Got encrypted AES196 RSA-OAEP-MGF1P SAML2 Assertion");

$decrypter = XML::Enc->new(
    {
        key                 => 't/sign-private.pem',
        no_xml_declaration  => 1
    }
);

ok($decrypter->decrypt($xml) =~ /NETSAML2_70fbdf22f456/, "Successfully Decrypted AES196 RSA-OAEP-MGF1P SAML2 Assertion");

$base64 = <<'SAMLAES256_RSA-OAEP-MGF1P';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SAMLAES256_RSA-OAEP-MGF1P

$xml = decode_base64($base64);

ok($xml, "Got encrypted AES256 RSA-OAEP-MGF1P SAML2 Assertion");

$decrypter = XML::Enc->new(
    {
        key                 => 't/sign-private.pem',
        no_xml_declaration  => 1
    }
);
ok($decrypter->decrypt($xml) =~ /A835657d0615aa0bfa/, "Successfully Decrypted AES256 RSA-OAEP-MGF1P SAML2 Assertion");

done_testing;
