/*--------------------------------------------------------------------------*/
/* ALBERTA:  an Adaptive multi Level finite element toolbox using           */
/*           Bisectioning refinement and Error control by Residual          */
/*           Techniques for scientific Applications                         */
/*                                                                          */
/* file:     lagrange_3_2d.c                                                */
/*                                                                          */
/* description:  piecewise cubic Lagrange elements in 2d                    */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/*  authors:   Alfred Schmidt                                               */
/*             Zentrum fuer Technomathematik                                */
/*             Fachbereich 3 Mathematik/Informatik                          */
/*             Universitaet Bremen                                          */
/*             Bibliothekstr. 2                                             */
/*             D-28359 Bremen, Germany                                      */
/*                                                                          */
/*             Kunibert G. Siebert                                          */
/*             Institut fuer Mathematik                                     */
/*             Universitaet Augsburg                                        */
/*             Universitaetsstr. 14                                         */
/*             D-86159 Augsburg, Germany                                    */
/*                                                                          */
/*  http://www.mathematik.uni-freiburg.de/IAM/ALBERTA                       */
/*                                                                          */
/*  (c) by A. Schmidt and K.G. Siebert (1996-2003)                          */
/*                                                                          */
/*--------------------------------------------------------------------------*/

#define N_BAS3_2D  10

static const REAL bary3_2d[N_BAS3_2D][N_LAMBDA] = {{1.0, 0.0, 0.0, 0.0},
					  {0.0, 1.0, 0.0, 0.0},
					  {0.0, 0.0, 1.0, 0.0},
					  {0.0, 2.0/3.0, 1.0/3.0, 0.0},
					  {0.0, 1.0/3.0, 2.0/3.0, 0.0},
					  {1.0/3.0, 0.0, 2.0/3.0, 0.0},
					  {2.0/3.0, 0.0, 1.0/3.0, 0.0},
					  {2.0/3.0, 1.0/3.0, 0.0, 0.0},
					  {1.0/3.0, 2.0/3.0, 0.0, 0.0},
					  {1.0/3.0, 1.0/3.0, 1.0/3.0, 0.0}};

/*--------------------------------------------------------------------------*/
/*  basisfunction at vertex 0                                               */
/*--------------------------------------------------------------------------*/

static REAL phi3v0_2d(const REAL lambda[N_LAMBDA])
{
  return((4.5*(lambda[0] - 1.0)*lambda[0] + 1.0)*lambda[0]);
}

static const REAL *grd_phi3v0_2d(const REAL lambda[N_LAMBDA])
{
  static REAL grd[N_LAMBDA] = {0};
  grd[0] = (13.5*lambda[0] - 9.0)*lambda[0] + 1;
  return((const REAL *) grd);
}

static const REAL (*D2_phi3v0_2d(const REAL *lambda))[N_LAMBDA]
{
  static  REAL D2[N_LAMBDA][N_LAMBDA];
  D2[0][0] = 27.0*lambda[0] - 9.0;
  return((const REAL (*)[N_LAMBDA]) D2);
}

/*--------------------------------------------------------------------------*/
/*  basisfunction at vertex 1                                               */
/*--------------------------------------------------------------------------*/

static REAL phi3v1_2d(const REAL lambda[N_LAMBDA])
{
  return((4.5*(lambda[1] - 1.0)*lambda[1] + 1.0)*lambda[1]);
}

static const REAL *grd_phi3v1_2d(const REAL lambda[N_LAMBDA])
{
  static REAL grd[N_LAMBDA] = {0};
  grd[1] = (13.5*lambda[1] - 9.0)*lambda[1] + 1;
  return((const REAL *) grd);
}

static const REAL (*D2_phi3v1_2d(const REAL *lambda))[N_LAMBDA]
{
  static  REAL D2[N_LAMBDA][N_LAMBDA];
  D2[1][1] = 27.0*lambda[1] - 9.0;
  return((const REAL (*)[N_LAMBDA]) D2);
}

/*--------------------------------------------------------------------------*/
/*  basisfunction at vertex 2                                               */
/*--------------------------------------------------------------------------*/

static REAL phi3v2_2d(const REAL lambda[N_LAMBDA])
{
  return((4.5*(lambda[2] - 1.0)*lambda[2] + 1.0)*lambda[2]);
}

static const REAL *grd_phi3v2_2d(const REAL lambda[N_LAMBDA])
{
  static REAL grd[N_LAMBDA] = {0};
  grd[2] = (13.5*lambda[2] - 9.0)*lambda[2] + 1;
  return((const REAL *) grd);
}

static const REAL (*D2_phi3v2_2d(const REAL *lambda))[N_LAMBDA]
{
  static  REAL D2[N_LAMBDA][N_LAMBDA];
  D2[2][2] = 27.0*lambda[2] - 9.0;
  return((const REAL (*)[N_LAMBDA]) D2);
}

/*--------------------------------------------------------------------------*/
/*  basisfunction 0 at edge 0                                               */
/*--------------------------------------------------------------------------*/

static REAL phi3e0_2d(const REAL lambda[N_LAMBDA])
{
  return((13.5*lambda[1] - 4.5)*lambda[1]*lambda[2]);
}

static const REAL *grd_phi3e0_2d(const REAL lambda[N_LAMBDA])
{
  static REAL grd[N_LAMBDA] = {0};
  grd[1] = (27.0*lambda[1] - 4.5)*lambda[2];
  grd[2] = (13.5*lambda[1] - 4.5)*lambda[1];
  return((const REAL *) grd);
}

static const REAL (*D2_phi3e0_2d(const REAL *lambda))[N_LAMBDA]
{
  static  REAL D2[N_LAMBDA][N_LAMBDA];
  D2[1][1] = 27.0*lambda[2];
  D2[1][2] = D2[2][1] = 27.0*lambda[1] - 4.5;
  return((const REAL (*)[N_LAMBDA]) D2);
}

/*--------------------------------------------------------------------------*/
/*  basisfunction 1 at edge 0                                               */
/*--------------------------------------------------------------------------*/

static REAL phi3e1_2d(const REAL lambda[N_LAMBDA])
{
  return((13.5*lambda[2] - 4.5)*lambda[2]*lambda[1]);
}

static const REAL *grd_phi3e1_2d(const REAL lambda[N_LAMBDA])
{
  static REAL grd[N_LAMBDA] = {0};
  grd[1] = (13.5*lambda[2] - 4.5)*lambda[2];
  grd[2] = (27.0*lambda[2] - 4.5)*lambda[1];
  return((const REAL *) grd);
}

static const REAL (*D2_phi3e1_2d(const REAL *lambda))[N_LAMBDA]
{
  static  REAL D2[N_LAMBDA][N_LAMBDA];
  D2[1][2] = D2[2][1] = 27.0*lambda[2] - 4.5;
  D2[2][2] = 27.0*lambda[1];
  return((const REAL (*)[N_LAMBDA]) D2);
}

/*--------------------------------------------------------------------------*/
/*  basisfunction 0 at edge 1                                               */
/*--------------------------------------------------------------------------*/

static REAL phi3e2_2d(const REAL lambda[N_LAMBDA])
{
  return((13.5*lambda[2] - 4.5)*lambda[2]*lambda[0]);
}

static const REAL *grd_phi3e2_2d(const REAL lambda[N_LAMBDA])
{
  static REAL grd[N_LAMBDA] = {0};
  grd[0] = (13.5*lambda[2] - 4.5)*lambda[2];
  grd[2] = (27.0*lambda[2] - 4.5)*lambda[0];
  return((const REAL *) grd);
}

static const REAL (*D2_phi3e2_2d(const REAL *lambda))[N_LAMBDA]
{
  static  REAL D2[N_LAMBDA][N_LAMBDA];
  D2[0][2] = D2[2][0] = 27.0*lambda[2] - 4.5;
  D2[2][2] = 27.0*lambda[0];
  return((const REAL (*)[N_LAMBDA]) D2);
}

/*--------------------------------------------------------------------------*/
/*  basisfunction 1 at edge 1                                               */
/*--------------------------------------------------------------------------*/

static REAL phi3e3_2d(const REAL lambda[N_LAMBDA])
{
  return((13.5*lambda[0] - 4.5)*lambda[0]*lambda[2]);
}

static const REAL *grd_phi3e3_2d(const REAL lambda[N_LAMBDA])
{
  static REAL grd[N_LAMBDA] = {0};
  grd[0] = (27.0*lambda[0] - 4.5)*lambda[2];
  grd[2] = (13.5*lambda[0] - 4.5)*lambda[0];
  return((const REAL *) grd);
}

static const REAL (*D2_phi3e3_2d(const REAL *lambda))[N_LAMBDA]
{
  static  REAL D2[N_LAMBDA][N_LAMBDA];
  D2[0][0] = 27.0*lambda[2];
  D2[0][2] = D2[2][0] = 27.0*lambda[0] - 4.5;
  return((const REAL (*)[N_LAMBDA]) D2);
}

/*--------------------------------------------------------------------------*/
/*  basisfunction 0 at edge 2                                               */
/*--------------------------------------------------------------------------*/

static REAL phi3e4_2d(const REAL lambda[N_LAMBDA])
{
  return((13.5*lambda[0] - 4.5)*lambda[0]*lambda[1]);
}

static const REAL *grd_phi3e4_2d(const REAL lambda[N_LAMBDA])
{
  static REAL grd[N_LAMBDA] = {0};
  grd[0] = (27.0*lambda[0] - 4.5)*lambda[1];
  grd[1] = (13.5*lambda[0] - 4.5)*lambda[0];
  return((const REAL *) grd);
}

static const REAL (*D2_phi3e4_2d(const REAL *lambda))[N_LAMBDA]
{
  static  REAL D2[N_LAMBDA][N_LAMBDA];
  D2[0][0] = 27.0*lambda[1];
  D2[0][1] = D2[1][0] = 27.0*lambda[0] - 4.5;
  return((const REAL (*)[N_LAMBDA]) D2);
}

/*--------------------------------------------------------------------------*/
/*  basisfunction 1 at edge 2                                               */
/*--------------------------------------------------------------------------*/

static REAL phi3e5_2d(const REAL lambda[N_LAMBDA])
{
  return((13.5*lambda[1] - 4.5)*lambda[1]*lambda[0]);
}

static const REAL *grd_phi3e5_2d(const REAL lambda[N_LAMBDA])
{
  static REAL grd[N_LAMBDA] = {0};
  grd[0] = (13.5*lambda[1] - 4.5)*lambda[1];
  grd[1] = (27.0*lambda[1] - 4.5)*lambda[0];
  return((const REAL *) grd);
}

static const REAL (*D2_phi3e5_2d(const REAL *lambda))[N_LAMBDA]
{
  static  REAL D2[N_LAMBDA][N_LAMBDA];
  D2[0][1] = D2[1][0] = 27.0*lambda[1] - 4.5;
  D2[1][1] = 27.0*lambda[0];
  return((const REAL (*)[N_LAMBDA]) D2);
}

/*--------------------------------------------------------------------------*/
/*  basisfunction at barycenter                                             */
/*--------------------------------------------------------------------------*/

static REAL phi3c0_2d(const REAL lambda[N_LAMBDA])
{
  return(27.0*lambda[0]*lambda[1]*lambda[2]);
}

static const REAL *grd_phi3c0_2d(const REAL lambda[N_LAMBDA])
{
  static REAL grd[N_LAMBDA] = {0};
  grd[0] = 27.0*lambda[1]*lambda[2];
  grd[1] = 27.0*lambda[0]*lambda[2];
  grd[2] = 27.0*lambda[0]*lambda[1];
  return((const REAL *) grd);
}

static const REAL (*D2_phi3c0_2d(const REAL *lambda))[N_LAMBDA]
{
  static  REAL D2[N_LAMBDA][N_LAMBDA];
  D2[0][1] = D2[1][0] = 27.0*lambda[2];
  D2[0][2] = D2[2][0] = 27.0*lambda[1];
  D2[1][2] = D2[2][1] = 27.0*lambda[0];
  return((const REAL (*)[N_LAMBDA]) D2);
}

/*--------------------------------------------------------------------------*/
/*  functions for combining basisfunctions with coefficients                */
/*--------------------------------------------------------------------------*/


static const DOF *get_dof_indices3_2d(const EL *el, const DOF_ADMIN *admin,
				      DOF *idof)
{
  static DOF  index_vec[N_BAS3_2D];
  DOF         *rvec = idof ? idof : index_vec;
  int         i, j, n0;
  DOF         **dof = el->dof;

  n0 = admin->n0_dof[VERTEX];
  for (i = 0; i < N_VERTICES_2D; i++)
    rvec[i] = dof[i][n0];
  n0 = admin->n0_dof[EDGE];
  for (i = 0, j = N_VERTICES_2D; i < N_EDGES_2D; i++)
  {
    if (dof[vertex_of_edge_2d[i][0]][0] < dof[vertex_of_edge_2d[i][1]][0])
    {
      rvec[j++] = dof[N_VERTICES_2D+i][n0];
      rvec[j++] = dof[N_VERTICES_2D+i][n0+1];
    }
    else
    {
      rvec[j++] = dof[N_VERTICES_2D+i][n0+1];
      rvec[j++] = dof[N_VERTICES_2D+i][n0];
    }
  }
  n0 = admin->n0_dof[CENTER];
  rvec[9] = dof[6][n0];

  return((const DOF *) rvec);
}


static const S_CHAR *get_bound3_2d(const EL_INFO *el_info, S_CHAR *bound)
{
  FUNCNAME("get_bound3_2d");
  static S_CHAR  bound_vec[N_BAS3_2D];
  S_CHAR         *rvec = bound ? bound : bound_vec;
  int            i, j;

  TEST_FLAG(FILL_BOUND, el_info);

  for (i = 0; i < N_VERTICES_2D; i++)
    rvec[i] = el_info->vertex_bound[i];

  for (j = N_VERTICES_2D, i = 0; i < N_EDGES_2D; j += 2, i++)
    rvec[j] = rvec[j+1] = el_info->edge_bound[i];

  rvec[9] = INTERIOR;

  return((const S_CHAR *) rvec);
}

static const int *get_int_vec3_2d(const EL *el, const DOF_INT_VEC *vec,
				  int *ivec)
{
  FUNCNAME("get_int_vec3_2d");
  int            i, j, n0;
  static int     local_vec[N_BAS3_2D];
  int            *v = nil, *rvec = ivec ? ivec : local_vec;
  DOF            **dof = el->dof;

  GET_DOF_VEC(v, vec);

  n0 = vec->fe_space->admin->n0_dof[VERTEX];
  for (i = 0; i < N_VERTICES_2D; i++)
    rvec[i] = v[dof[i][n0]];
  n0 = vec->fe_space->admin->n0_dof[EDGE];
  for (i = 0, j = N_VERTICES_2D; i < N_EDGES_2D; i++)
  {
    if (dof[vertex_of_edge_2d[i][0]][0] < dof[vertex_of_edge_2d[i][1]][0])
    {
      rvec[j++] = v[dof[N_VERTICES_2D+i][n0]];
      rvec[j++] = v[dof[N_VERTICES_2D+i][n0+1]];
    }
    else
    {
      rvec[j++] = v[dof[N_VERTICES_2D+i][n0+1]];
      rvec[j++] = v[dof[N_VERTICES_2D+i][n0]];
    }
  }
  n0 = vec->fe_space->admin->n0_dof[CENTER];
  rvec[9] = v[dof[6][n0]];

  return((const int *) rvec);
}

static const REAL *get_real_vec3_2d(const EL *el, const DOF_REAL_VEC *vec,
				    REAL *Rvec)
{
  FUNCNAME("get_real_vec3_2d");
  int            i, j, n0;
  static REAL    local_vec[N_BAS3_2D];
  REAL           *v = nil, *rvec = Rvec ? Rvec : local_vec;
  DOF            **dof = el->dof;

  GET_DOF_VEC(v, vec);

  n0 = vec->fe_space->admin->n0_dof[VERTEX];
  for (i = 0; i < N_VERTICES_2D; i++)
    rvec[i] = v[dof[i][n0]];
  n0 = vec->fe_space->admin->n0_dof[EDGE];
  for (i = 0, j = N_VERTICES_2D; i < N_EDGES_2D; i++)
  {
    if (dof[vertex_of_edge_2d[i][0]][0] < dof[vertex_of_edge_2d[i][1]][0])
    {
      rvec[j++] = v[dof[N_VERTICES_2D+i][n0]];
      rvec[j++] = v[dof[N_VERTICES_2D+i][n0+1]];
    }
    else
    {
      rvec[j++] = v[dof[N_VERTICES_2D+i][n0+1]];
      rvec[j++] = v[dof[N_VERTICES_2D+i][n0]];
    }
  }
  n0 = vec->fe_space->admin->n0_dof[CENTER];
  rvec[9] = v[dof[6][n0]];

  return((const REAL *) rvec);
}

static const REAL_D *get_real_d_vec3_2d(const EL *el,
					const DOF_REAL_D_VEC *vec,
					REAL_D *Rvec)
{
  FUNCNAME("get_real_d_vec3_2d");
  int            i, j, k, n0;
  static REAL_D  local_vec[N_BAS3_2D];
  REAL_D         *v = nil, *rvec = Rvec ? Rvec : local_vec;
  DOF            **dof = el->dof;

  GET_DOF_VEC(v, vec);

  n0 = vec->fe_space->admin->n0_dof[VERTEX];
  for (i = 0; i < N_VERTICES_2D; i++)
    for (k = 0; k < DIM_OF_WORLD; k++)
      rvec[i][k] = v[dof[i][n0]][k];

  n0 = vec->fe_space->admin->n0_dof[EDGE];
  for (i = 0, j = N_VERTICES_2D; i < N_EDGES_2D; i++)
  {
    if (dof[vertex_of_edge_2d[i][0]][0] < dof[vertex_of_edge_2d[i][1]][0])
    {
      for (k = 0; k < DIM_OF_WORLD; k++)
      {
	rvec[j][k] = v[dof[N_VERTICES_2D+i][n0]][k];
	rvec[j+1][k] = v[dof[N_VERTICES_2D+i][n0+1]][k];
      }
      j += 2;
    }
    else
    {
      for (k = 0; k < DIM_OF_WORLD; k++)
      {
	rvec[j][k] = v[dof[N_VERTICES_2D+i][n0+1]][k];
	rvec[j+1][k] = v[dof[N_VERTICES_2D+i][n0]][k];
      }
      j += 2;
    }
  }
  n0 = vec->fe_space->admin->n0_dof[CENTER];
  for (k = 0; k < DIM_OF_WORLD; k++)
    rvec[9][k] = v[dof[6][n0]][k];

  return((const REAL_D *) rvec);
}

static const U_CHAR *get_uchar_vec3_2d(const EL *el, const DOF_UCHAR_VEC *vec,
				       U_CHAR *uvec)
{
  FUNCNAME("get_uchar_vec3_2d");
  int            i, j, n0;
  static U_CHAR  local_vec[N_BAS3_2D];
  U_CHAR         *v = nil, *rvec = uvec ? uvec : local_vec;
  DOF            **dof = el->dof;

  GET_DOF_VEC(v, vec);

  n0 = vec->fe_space->admin->n0_dof[VERTEX];
  for (i = 0; i < N_VERTICES_2D; i++)
    rvec[i] = v[dof[i][n0]];
  n0 = vec->fe_space->admin->n0_dof[EDGE];
  for (i = 0, j = N_VERTICES_2D; i < N_EDGES_2D; i++)
  {
    if (dof[vertex_of_edge_2d[i][0]][0] < dof[vertex_of_edge_2d[i][1]][0])
    {
      rvec[j++] = v[dof[N_VERTICES_2D+i][n0]];
      rvec[j++] = v[dof[N_VERTICES_2D+i][n0+1]];
    }
    else
    {
      rvec[j++] = v[dof[N_VERTICES_2D+i][n0+1]];
      rvec[j++] = v[dof[N_VERTICES_2D+i][n0]];
    }
  }
  n0 = vec->fe_space->admin->n0_dof[CENTER];
  rvec[9] = v[dof[6][n0]];

  return((const U_CHAR *) rvec);
}

static const S_CHAR *get_schar_vec3_2d(const EL *el, const DOF_SCHAR_VEC *vec,
				       S_CHAR *svec)
{
  FUNCNAME("get_schar_vec3_2d");
  int            i, j, n0;
  static S_CHAR  local_vec[N_BAS3_2D];
  S_CHAR         *v = nil, *rvec = svec ? svec : local_vec;
  DOF            **dof = el->dof;

  GET_DOF_VEC(v, vec);

  n0 = vec->fe_space->admin->n0_dof[VERTEX];
  for (i = 0; i < N_VERTICES_2D; i++)
    rvec[i] = v[dof[i][n0]];
  n0 = vec->fe_space->admin->n0_dof[EDGE];
  for (i = 0, j = N_VERTICES_2D; i < N_EDGES_2D; i++)
  {
    if (dof[vertex_of_edge_2d[i][0]][0] < dof[vertex_of_edge_2d[i][1]][0])
    {
      rvec[j++] = v[dof[N_VERTICES_2D+i][n0]];
      rvec[j++] = v[dof[N_VERTICES_2D+i][n0+1]];
    }
    else
    {
      rvec[j++] = v[dof[N_VERTICES_2D+i][n0+1]];
      rvec[j++] = v[dof[N_VERTICES_2D+i][n0]];
    }
  }
  n0 = vec->fe_space->admin->n0_dof[CENTER];
  rvec[9] = v[dof[6][n0]];

  return((const S_CHAR *) rvec);
}

/*--------------------------------------------------------------------*/
/*--- function for local interpolaton of scalar functions          ---*/
/*--------------------------------------------------------------------*/

GENERATE_INTERPOL(,3,2,10)

/*--------------------------------------------------------------------*/
/*--- function for local interpolaton of vector functions          ---*/
/*--------------------------------------------------------------------*/

GENERATE_INTERPOL_D(,3,2,10)
  
/*--------------------------------------------------------------------------*/
/*  functions for interpolation/ restriction during refinement/coarsening   */
/*--------------------------------------------------------------------------*/

static void real_refine_inter3_2d(DOF_REAL_VEC *drv, RC_LIST_EL *list, int n)
{
  FUNCNAME("real_refine_inter3_2d");
  EL        *el;
  REAL      *v = nil;
  int       dof9, node, n0;
  const DOF *cdof;
  DOF       pdof[N_BAS3_2D];
  const DOF       *(*get_dof_indices)(const EL *, const DOF_ADMIN *, DOF *);
  const DOF_ADMIN *admin;

  if (n < 1) return;
  el = list->el_info.el;

  GET_DOF_VEC(v, drv);
  if (!drv->fe_space)
  {
    ERROR("no fe_space in dof_real_vec %s\n", NAME(drv));
    return;
  }
  else if (!drv->fe_space->bas_fcts)
  {
    ERROR("no basis functions in fe_space %s\n", NAME(drv->fe_space));
    return;
  }
  get_dof_indices = drv->fe_space->bas_fcts->get_dof_indices;
  GET_STRUCT(admin,drv->fe_space);

  get_dof_indices(el, admin, pdof);
  
/*--------------------------------------------------------------------------*/
/*  values on child[0]                                                      */
/*--------------------------------------------------------------------------*/
  cdof = get_dof_indices(el->child[0], admin, nil);

  v[cdof[2]] = (-0.0625*(v[pdof[0]] + v[pdof[1]]) 
		+ 0.5625*(v[pdof[7]] + v[pdof[8]]));
  v[cdof[3]] = (0.3125*(v[pdof[0]] - v[pdof[8]]) + 0.0625*v[pdof[1]]
		+ 0.9375*v[pdof[7]]);
  v[cdof[4]] =  v[pdof[7]];
  v[cdof[5]] =  v[pdof[9]];
  v[cdof[6]] =  (0.0625*(v[pdof[0]] + v[pdof[1]])
		 - 0.25*(v[pdof[3]] + v[pdof[6]])
		 + 0.5*(v[pdof[4]] + v[pdof[5]] + v[pdof[9]])
		 - 0.0625*(v[pdof[7]] + v[pdof[8]]));
  v[cdof[9]] = (0.0625*(-v[pdof[0]] + v[pdof[1]]) - 0.125*v[pdof[3]]
		+ 0.375*v[pdof[6]] + 0.1875*(v[pdof[7]] - v[pdof[8]])
		+ 0.75*v[pdof[9]]);

/*--------------------------------------------------------------------------*/
/*  values on child[0]                                                      */
/*--------------------------------------------------------------------------*/

  cdof = get_dof_indices(el->child[1], admin, nil);

  v[cdof[5]] =  v[pdof[8]];
  v[cdof[6]] =  (0.0625*v[pdof[0]] + 0.9375*v[pdof[8]]
		 + 0.3125*(v[pdof[1]] - v[pdof[7]]));
  v[cdof[9]] =  (0.0625*(v[pdof[0]] - v[pdof[1]]) + 0.375*v[pdof[3]]
		 - 0.125*v[pdof[6]] + 0.1875*(-v[pdof[7]] + v[pdof[8]])
		 + 0.75*v[pdof[9]]);

  if (n <= 1)  return;

/*--------------------------------------------------------------------------*/
/*  adjust the value on the neihgbour                                       */
/*--------------------------------------------------------------------------*/

  el = list[1].el_info.el;
  get_dof_indices(el, admin, pdof);

/*--------------------------------------------------------------------------*/
/*  values on neigh's child[0]                                              */
/*--------------------------------------------------------------------------*/
  cdof = get_dof_indices(el->child[0], admin, nil);
  
  v[cdof[5]] =  v[pdof[9]];
  v[cdof[6]] =  (0.0625*(v[pdof[0]] + v[pdof[1]]) 
		 - 0.25*(v[pdof[3]] + v[pdof[6]])
		 + 0.5*(v[pdof[4]] + v[pdof[5]] + v[pdof[9]])
		 - 0.0625*(v[pdof[7]] + v[pdof[8]]));
  v[cdof[9]] = (0.0625*(-v[pdof[0]] + v[pdof[1]]) - 0.12500*v[pdof[3]]
		+ 0.375*v[pdof[6]] + 0.1875*(v[pdof[7]] - v[pdof[8]])
		+ 0.75*v[pdof[9]]);
/*--------------------------------------------------------------------------*/
/*  values on neigh's child[0]                                              */
/*--------------------------------------------------------------------------*/

  node = drv->fe_space->mesh->node[CENTER];  
  n0 = admin->n0_dof[CENTER];
  dof9 = el->child[1]->dof[node][n0];

  v[dof9] =  (0.0625*(v[pdof[0]] - v[pdof[1]]) +  0.375*v[pdof[3]]
	      - 0.125*v[pdof[6]] + 0.1875*(-v[pdof[7]] + v[pdof[8]])
	      + 0.75*v[pdof[9]]);
  return;
}

static void real_coarse_inter3_2d(DOF_REAL_VEC *drv, RC_LIST_EL *list, int n)
{
  FUNCNAME("real_coarse_inter3_2d");
  EL        *el, *child;
  REAL      *v = nil;
  int       cdof, pdof, node, n0;
  const DOF_ADMIN *admin;
  MESH      *mesh = nil;

  if (n < 1) return;
  el = list->el_info.el;

  GET_DOF_VEC(v, drv);
  if (!drv->fe_space)
  {
    ERROR("no fe_space in dof_real_vec %s\n", NAME(drv));
    return;
  }
  else if (!drv->fe_space->bas_fcts)
  {
    ERROR("no basis functions in fe_space %s\n", NAME(drv->fe_space));
    return;
  }
  GET_STRUCT(admin,drv->fe_space);
  GET_STRUCT(mesh,drv->fe_space);

  node = mesh->node[EDGE];        
  n0 = admin->n0_dof[EDGE];

/*--------------------------------------------------------------------------*/
/*  values on child[0]                                                      */
/*--------------------------------------------------------------------------*/

  child = el->child[0];

  if (el->dof[0][0] < el->dof[1][0])
    pdof = el->dof[node+2][n0];
  else
    pdof = el->dof[node+2][n0+1];

  if (child->dof[1][0] < child->dof[2][0])
    cdof = child->dof[node][n0+1];
  else
    cdof = child->dof[node][n0];
      
  v[pdof] = v[cdof];

  if (child->dof[2][0] < child->dof[0][0])
    cdof = child->dof[node+1][n0];
  else
    cdof = child->dof[node+1][n0+1];
      
  v[el->dof[mesh->node[CENTER]][admin->n0_dof[CENTER]]] = v[cdof];

/*--------------------------------------------------------------------------*/
/*  values on child[1]                                                      */
/*--------------------------------------------------------------------------*/
  
  child = el->child[1];

  if (el->dof[0][0] < el->dof[1][0])
    pdof = el->dof[node+2][n0+1];
  else
    pdof = el->dof[node+2][n0];

  if (child->dof[2][0] < child->dof[0][0])
    cdof = child->dof[node+1][n0];
  else
    cdof = child->dof[node+1][n0+1];

  v[pdof] = v[cdof];

  if (n <= 1)  return;

/*--------------------------------------------------------------------------*/
/*   adjust neighbour values                                                */
/*--------------------------------------------------------------------------*/

  el = list[1].el_info.el;
  child = el->child[0];

  if (child->dof[2][0] < child->dof[0][0])
    cdof = child->dof[node+1][n0];
  else
    cdof = child->dof[node+1][n0+1];
      
  v[el->dof[mesh->node[CENTER]][admin->n0_dof[CENTER]]] = v[cdof];

  return;
}

static void real_coarse_restr3_2d(DOF_REAL_VEC *drv, RC_LIST_EL *list, int n)
{
  FUNCNAME("real_coarse_restr3_2d");
  EL        *el;
  REAL      *v = nil;
  int       node, n0;
  const DOF *cdof;
  DOF       pdof[N_BAS3_2D], dof9;
  const DOF       *(*get_dof_indices)(const EL *, const DOF_ADMIN *, DOF *);
  const DOF_ADMIN *admin;

  if (n < 1) return;
  el = list->el_info.el;

  GET_DOF_VEC(v, drv);
  if (!drv->fe_space)
  {
    ERROR("no fe_space in dof_real_vec %s\n", NAME(drv));
    return;
  }
  else if (!drv->fe_space->bas_fcts)
  {
    ERROR("no basis functions in fe_space %s\n", NAME(drv->fe_space));
    return;
  }
  get_dof_indices = drv->fe_space->bas_fcts->get_dof_indices;
  GET_STRUCT(admin,drv->fe_space);

  get_dof_indices(el, admin, pdof);
  
/*--------------------------------------------------------------------------*/
/*  values on child[0]                                                      */
/*--------------------------------------------------------------------------*/
  cdof = get_dof_indices(el->child[0], admin, nil);

  v[pdof[0]] += (0.0625*(-v[cdof[2]] + v[cdof[6]] - v[cdof[9]])
		 + 0.3125*v[cdof[3]]);
  v[pdof[1]] += 0.0625*(-v[cdof[2]] +  v[cdof[3]] + v[cdof[6]] + v[cdof[9]]);
  v[pdof[3]] += -0.25*v[cdof[6]] - 0.125*v[cdof[9]];
  v[pdof[4]] +=  0.5*v[cdof[6]];
  v[pdof[5]] +=  0.5*v[cdof[6]];
  v[pdof[6]] += -0.25*v[cdof[6]] + 0.375*v[cdof[9]];
  v[pdof[7]] = (0.5625*v[cdof[2]] + 0.9375*v[cdof[3]] + v[cdof[4]] 
		- 0.0625*v[cdof[6]] + 0.1875*v[cdof[9]]);
  v[pdof[8]] = (0.5625*v[cdof[2]] - 0.3125*v[cdof[3]] 
		- 0.0625*v[cdof[6]] - 0.1875*v[cdof[9]]);
  v[pdof[9]] =  v[cdof[5]] + 0.5*v[cdof[6]] + 0.75*v[cdof[9]];

/*--------------------------------------------------------------------------*/
/*  values on child[1]                                                      */
/*--------------------------------------------------------------------------*/

  cdof = get_dof_indices(el->child[1], admin, nil);

  v[pdof[0]] += 0.0625*v[cdof[6]] + 0.0625*v[cdof[9]];
  v[pdof[1]] += 0.3125*v[cdof[6]] - 0.0625*v[cdof[9]];
  v[pdof[3]] += 0.375*v[cdof[9]];
  v[pdof[6]] += -0.125*v[cdof[9]];
  v[pdof[7]] += -0.3125*v[cdof[6]] - 0.18750*v[cdof[9]];
  v[pdof[8]] += v[cdof[5]] + 0.9375*v[cdof[6]] + 0.1875*v[cdof[9]];
  v[pdof[9]] += 0.75*v[cdof[9]];

  if (n <= 1)  return;

/*--------------------------------------------------------------------------*/
/*  adjust the value on the neihgbour                                       */
/*--------------------------------------------------------------------------*/

  el = list[1].el_info.el;
  get_dof_indices(el, admin, pdof);

/*--------------------------------------------------------------------------*/
/*  values on neigh's child[0]                                              */
/*--------------------------------------------------------------------------*/
  cdof = get_dof_indices(el->child[0], admin, nil);
  
  v[pdof[0]] += 0.0625*(v[cdof[6]] - v[cdof[9]]);
  v[pdof[1]] += 0.0625*(v[cdof[6]] + v[cdof[9]]);
  v[pdof[3]] += -0.25*v[cdof[6]] - 0.12500*v[cdof[9]];
  v[pdof[4]] += 0.5*v[cdof[6]];
  v[pdof[5]] += 0.5*v[cdof[6]];
  v[pdof[6]] += -0.25*v[cdof[6]] + 0.375*v[cdof[9]];
  v[pdof[7]] += -0.0625*v[cdof[6]] + 0.1875*v[cdof[9]];
  v[pdof[8]] += -0.0625*v[cdof[6]] - 0.1875*v[cdof[9]];
  v[pdof[9]] =  v[cdof[5]] + 0.5*v[cdof[6]] + 0.75*v[cdof[9]];

/*--------------------------------------------------------------------------*/
/*  values on neigh's child[1]                                              */
/*--------------------------------------------------------------------------*/

  node = drv->fe_space->mesh->node[CENTER];  
  n0 = admin->n0_dof[CENTER];
  dof9 = el->child[1]->dof[node][n0];

  v[pdof[0]] += 0.0625*v[dof9];
  v[pdof[1]] -= 0.0625*v[dof9];
  v[pdof[3]] += 0.375*v[dof9];
  v[pdof[6]] -= 0.125*v[dof9];
  v[pdof[7]] -= 0.1875*v[dof9];
  v[pdof[8]] += 0.1875*v[dof9];
  v[pdof[9]] += 0.75*v[dof9];

  return;
}

static void real_d_refine_inter3_2d(DOF_REAL_D_VEC *drdv,
				    RC_LIST_EL *list, int n)
{
  FUNCNAME("real_d_refine_inter3_2d");
  EL        *el;
  REAL_D    *v = nil;
  int       node, n0, j;
  const DOF *cdof;
  DOF       pdof[N_BAS3_2D], dof9;
  const DOF       *(*get_dof_indices)(const EL *, const DOF_ADMIN *, DOF *);
  const DOF_ADMIN *admin;

  if (n < 1) return;
  el = list->el_info.el;

  GET_DOF_VEC(v, drdv);
  if (!drdv->fe_space)
  {
    ERROR("no fe_space in dof_real_d_vec %s\n", NAME(drdv));
    return;
  }
  else if (!drdv->fe_space->bas_fcts)
  {
    ERROR("no basis functions in fe_space %s\n", NAME(drdv->fe_space));
    return;
  }
  get_dof_indices = drdv->fe_space->bas_fcts->get_dof_indices;
  GET_STRUCT(admin,drdv->fe_space);

  get_dof_indices(el, admin, pdof);
  
/*--------------------------------------------------------------------------*/
/*  values on child[0]                                                      */
/*--------------------------------------------------------------------------*/
  cdof = get_dof_indices(el->child[0], admin, nil);

  for (j = 0; j < DIM_OF_WORLD; j++)
  {
    v[cdof[2]][j] = (-0.0625*(v[pdof[0]][j] + v[pdof[1]][j]) 
		     + 0.5625*(v[pdof[7]][j] + v[pdof[8]][j]));
    v[cdof[3]][j] = (0.3125*(v[pdof[0]][j] - v[pdof[8]][j])
		     + 0.0625*v[pdof[1]][j] + 0.9375*v[pdof[7]][j]);
    v[cdof[4]][j] =  v[pdof[7]][j];
    v[cdof[5]][j] =  v[pdof[9]][j];
    v[cdof[6]][j] =  (0.0625*(v[pdof[0]][j] + v[pdof[1]][j])
		      - 0.25*(v[pdof[3]][j] + v[pdof[6]][j])
		      + 0.5*(v[pdof[4]][j] + v[pdof[5]][j] + v[pdof[9]][j])
		      - 0.0625*(v[pdof[7]][j] + v[pdof[8]][j]));
    v[cdof[9]][j] = (0.0625*(-v[pdof[0]][j] + v[pdof[1]][j])
		     - 0.125*v[pdof[3]][j] + 0.375*v[pdof[6]][j]
		     + 0.1875*(v[pdof[7]][j] - v[pdof[8]][j])
		     + 0.75*v[pdof[9]][j]);
  }

/*--------------------------------------------------------------------------*/
/*  values on child[0]                                                      */
/*--------------------------------------------------------------------------*/

  cdof = get_dof_indices(el->child[1], admin, nil);

  for (j = 0; j < DIM_OF_WORLD; j++)
  {
    v[cdof[5]][j] =  v[pdof[8]][j];
    v[cdof[6]][j] =  (0.0625*v[pdof[0]][j] + 0.9375*v[pdof[8]][j]
		      + 0.3125*(v[pdof[1]][j] - v[pdof[7]][j]));
    v[cdof[9]][j] =  (0.0625*(v[pdof[0]][j] - v[pdof[1]][j]) 
		      + 0.375*v[pdof[3]][j] - 0.125*v[pdof[6]][j]
		      + 0.1875*(-v[pdof[7]][j] + v[pdof[8]][j])
		      + 0.75*v[pdof[9]][j]);
  }

  if (n <= 1)  return;

/*--------------------------------------------------------------------------*/
/*  adjust the value on the neihgbour                                       */
/*--------------------------------------------------------------------------*/

  el = list[1].el_info.el;
  get_dof_indices(el, admin, pdof);

/*--------------------------------------------------------------------------*/
/*  values on neigh's child[0]                                              */
/*--------------------------------------------------------------------------*/
  cdof = get_dof_indices(el->child[0], admin, nil);
  

  for (j = 0; j < DIM_OF_WORLD; j++)
  {
    v[cdof[5]][j] =  v[pdof[9]][j];
    v[cdof[6]][j] =  (0.0625*(v[pdof[0]][j] + v[pdof[1]][j]) 
		      - 0.25*(v[pdof[3]][j] + v[pdof[6]][j])
		      + 0.5*(v[pdof[4]][j] + v[pdof[5]][j] + v[pdof[9]][j])
		      - 0.0625*(v[pdof[7]][j] + v[pdof[8]][j]));
    v[cdof[9]][j] = (0.0625*(-v[pdof[0]][j] + v[pdof[1]][j])
		     - 0.12500*v[pdof[3]][j] + 0.375*v[pdof[6]][j]
		     + 0.1875*(v[pdof[7]][j] - v[pdof[8]][j])
		     + 0.75*v[pdof[9]][j]);
  }

/*--------------------------------------------------------------------------*/
/*  values on neigh's child[0]                                              */
/*--------------------------------------------------------------------------*/

  node = drdv->fe_space->mesh->node[CENTER];  
  n0 = admin->n0_dof[CENTER];

  for (j = 0; j < DIM_OF_WORLD; j++)
  {
    dof9 = el->child[1]->dof[node][n0];

    v[dof9][j] =  (0.0625*(v[pdof[0]][j] - v[pdof[1]][j])
		   +  0.375*v[pdof[3]][j] - 0.125*v[pdof[6]][j]
		   + 0.1875*(-v[pdof[7]][j] + v[pdof[8]][j])
		   + 0.75*v[pdof[9]][j]);
  }
  return;
}

static void real_d_coarse_inter3_2d(DOF_REAL_D_VEC *drdv,
				    RC_LIST_EL *list, int n)
{
  FUNCNAME("real_d_coarse_inter3_2d");
  EL        *el, *child;
  REAL_D    *v = nil;
  int       cdof, pdof, node, n0, j;
  const DOF_ADMIN *admin;
  MESH      *mesh = nil;

  if (n < 1) return;
  el = list->el_info.el;

  GET_DOF_VEC(v, drdv);
  if (!drdv->fe_space)
  {
    ERROR("no fe_space in dof_real_d_vec %s\n", NAME(drdv));
    return;
  }
  else if (!drdv->fe_space->bas_fcts)
  {
    ERROR("no basis functions in fe_space %s\n", NAME(drdv->fe_space));
    return;
  }
  GET_STRUCT(admin,drdv->fe_space);
  GET_STRUCT(mesh,drdv->fe_space);

  node = mesh->node[EDGE];        
  n0 = admin->n0_dof[EDGE];

/*--------------------------------------------------------------------------*/
/*  values on child[0]                                                      */
/*--------------------------------------------------------------------------*/

  child = el->child[0];

  if (el->dof[0][0] < el->dof[1][0])
    pdof = el->dof[node+2][n0];
  else
    pdof = el->dof[node+2][n0+1];

  if (child->dof[1][0] < child->dof[2][0])
    cdof = child->dof[node][n0+1];
  else
    cdof = child->dof[node][n0];

  for (j = 0; j < DIM_OF_WORLD; j++)
    v[pdof][j] = v[cdof][j];

  if (child->dof[2][0] < child->dof[0][0])
    cdof = child->dof[node+1][n0];
  else
    cdof = child->dof[node+1][n0+1];
      
  for (j = 0; j < DIM_OF_WORLD; j++)
    v[el->dof[mesh->node[CENTER]][admin->n0_dof[CENTER]]][j] = v[cdof][j];

/*--------------------------------------------------------------------------*/
/*  values on child[1]                                                      */
/*--------------------------------------------------------------------------*/
  
  child = el->child[1];

  if (el->dof[0][0] < el->dof[1][0])
    pdof = el->dof[node+2][n0+1];
  else
    pdof = el->dof[node+2][n0];

  if (child->dof[2][0] < child->dof[0][0])
    cdof = child->dof[node+1][n0];
  else
    cdof = child->dof[node+1][n0+1];

  for (j = 0; j < DIM_OF_WORLD; j++)
    v[pdof][j] = v[cdof][j];

  if (n <= 1)  return;

/*--------------------------------------------------------------------------*/
/*   adjust neighbour values                                                */
/*--------------------------------------------------------------------------*/

  el = list[1].el_info.el;
  child = el->child[0];

  if (child->dof[2][0] < child->dof[0][0])
    cdof = child->dof[node+1][n0];
  else
    cdof = child->dof[node+1][n0+1];
      
  for (j = 0; j < DIM_OF_WORLD; j++)
    v[el->dof[mesh->node[CENTER]][admin->n0_dof[CENTER]]][j] = v[cdof][j];

  return;
}

static void real_d_coarse_restr3_2d(DOF_REAL_D_VEC *drdv,
				    RC_LIST_EL *list, int n)
{
  FUNCNAME("real_d_coarse_restr3_2d");
  EL        *el;
  REAL_D    *v = nil;
  int       node, n0, j;
  const DOF *cdof;
  DOF       pdof[N_BAS3_2D], dof9;
  const DOF       *(*get_dof_indices)(const EL *, const DOF_ADMIN *, DOF *);
  const DOF_ADMIN *admin;

  if (n < 1) return;
  el = list->el_info.el;

  GET_DOF_VEC(v, drdv);
  if (!drdv->fe_space)
  {
    ERROR("no fe_space in dof_real_d_vec %s\n", NAME(drdv));
    return;
  }
  else if (!drdv->fe_space->bas_fcts)
  {
    ERROR("no basis functions in fe_space %s\n", NAME(drdv->fe_space));
    return;
  }
  get_dof_indices = drdv->fe_space->bas_fcts->get_dof_indices;
  GET_STRUCT(admin,drdv->fe_space);

  get_dof_indices(el, admin, pdof);
  
/*--------------------------------------------------------------------------*/
/*  values on child[0]                                                      */
/*--------------------------------------------------------------------------*/
  cdof = get_dof_indices(el->child[0], admin, nil);

  for (j = 0; j < DIM_OF_WORLD; j++)
  {
    v[pdof[0]][j] += (0.0625*(-v[cdof[2]][j] + v[cdof[6]][j] - v[cdof[9]][j])
		      + 0.3125*v[cdof[3]][j]);
    v[pdof[1]][j] += 0.0625*(-v[cdof[2]][j] +  v[cdof[3]][j] + v[cdof[6]][j]
			     + v[cdof[9]][j]);
    v[pdof[3]][j] += -0.25*v[cdof[6]][j] - 0.125*v[cdof[9]][j];
    v[pdof[4]][j] +=  0.5*v[cdof[6]][j];
    v[pdof[5]][j] +=  0.5*v[cdof[6]][j];
    v[pdof[6]][j] += -0.25*v[cdof[6]][j] + 0.375*v[cdof[9]][j];
    v[pdof[7]][j] = (0.5625*v[cdof[2]][j] + 0.9375*v[cdof[3]][j]
		     + v[cdof[4]][j] - 0.0625*v[cdof[6]][j]
		     + 0.1875*v[cdof[9]][j]);
    v[pdof[8]][j] = (0.5625*v[cdof[2]][j] - 0.3125*v[cdof[3]][j] 
		     - 0.0625*v[cdof[6]][j] - 0.1875*v[cdof[9]][j]);
    v[pdof[9]][j] =  v[cdof[5]][j] + 0.5*v[cdof[6]][j] + 0.75*v[cdof[9]][j];
  }

/*--------------------------------------------------------------------------*/
/*  values on child[1]                                                      */
/*--------------------------------------------------------------------------*/

  cdof = get_dof_indices(el->child[1], admin, nil);

  for (j = 0; j < DIM_OF_WORLD; j++)
  {
    v[pdof[0]][j] += 0.0625*v[cdof[6]][j] + 0.0625*v[cdof[9]][j];
    v[pdof[1]][j] += 0.3125*v[cdof[6]][j] - 0.0625*v[cdof[9]][j];
    v[pdof[3]][j] += 0.375*v[cdof[9]][j];
    v[pdof[6]][j] += -0.125*v[cdof[9]][j];
    v[pdof[7]][j] += -0.3125*v[cdof[6]][j] - 0.18750*v[cdof[9]][j];
    v[pdof[8]][j] += (v[cdof[5]][j] + 0.9375*v[cdof[6]][j]
		      + 0.1875*v[cdof[9]][j]);
    v[pdof[9]][j] += 0.75*v[cdof[9]][j];
  }

  if (n <= 1)  return;

/*--------------------------------------------------------------------------*/
/*  adjust the value on the neihgbour                                       */
/*--------------------------------------------------------------------------*/

  el = list[1].el_info.el;
  get_dof_indices(el, admin, pdof);

/*--------------------------------------------------------------------------*/
/*  values on neigh's child[0]                                              */
/*--------------------------------------------------------------------------*/
  cdof = get_dof_indices(el->child[0], admin, nil);
  
  for (j = 0; j < DIM_OF_WORLD; j++)
  {
    v[pdof[0]][j] += 0.0625*(v[cdof[6]][j] - v[cdof[9]][j]);
    v[pdof[1]][j] += 0.0625*(v[cdof[6]][j] + v[cdof[9]][j]);
    v[pdof[3]][j] += -0.25*v[cdof[6]][j] - 0.12500*v[cdof[9]][j];
    v[pdof[4]][j] += 0.5*v[cdof[6]][j];
    v[pdof[5]][j] += 0.5*v[cdof[6]][j];
    v[pdof[6]][j] += -0.25*v[cdof[6]][j] + 0.375*v[cdof[9]][j];
    v[pdof[7]][j] += -0.0625*v[cdof[6]][j] + 0.1875*v[cdof[9]][j];
    v[pdof[8]][j] += -0.0625*v[cdof[6]][j] - 0.1875*v[cdof[9]][j];
    v[pdof[9]][j] =  v[cdof[5]][j] + 0.5*v[cdof[6]][j] + 0.75*v[cdof[9]][j];
  }

/*--------------------------------------------------------------------------*/
/*  values on neigh's child[1]                                              */
/*--------------------------------------------------------------------------*/

  node = drdv->fe_space->mesh->node[CENTER];  
  n0 = admin->n0_dof[CENTER];
  dof9 = el->child[1]->dof[node][n0];

  for (j = 0; j < DIM_OF_WORLD; j++)
  {
    v[pdof[0]][j] += 0.0625*v[dof9][j];
    v[pdof[1]][j] -= 0.0625*v[dof9][j];
    v[pdof[3]][j] += 0.375*v[dof9][j];
    v[pdof[6]][j] -= 0.125*v[dof9][j];
    v[pdof[7]][j] -= 0.1875*v[dof9][j];
    v[pdof[8]][j] += 0.1875*v[dof9][j];
    v[pdof[9]][j] += 0.75*v[dof9][j];
  }

  return;
}

static BAS_FCT      *phi3_2d[N_BAS3_2D] = {phi3v0_2d, phi3v1_2d, phi3v2_2d,
					   phi3e0_2d, phi3e1_2d, phi3e2_2d,
					   phi3e3_2d, phi3e4_2d, phi3e5_2d,
					   phi3c0_2d};
static GRD_BAS_FCT  *grd_phi3_2d[N_BAS3_2D] = {grd_phi3v0_2d, grd_phi3v1_2d, 
					       grd_phi3v2_2d, grd_phi3e0_2d,
					       grd_phi3e1_2d, grd_phi3e2_2d,
					       grd_phi3e3_2d, grd_phi3e4_2d,
					       grd_phi3e5_2d, grd_phi3c0_2d};
static D2_BAS_FCT   *D2_phi3_2d[N_BAS3_2D]  = {D2_phi3v0_2d, D2_phi3v1_2d, 
					       D2_phi3v2_2d, D2_phi3e0_2d,
					       D2_phi3e1_2d, D2_phi3e2_2d,
					       D2_phi3e3_2d, D2_phi3e4_2d,
					       D2_phi3e5_2d, D2_phi3c0_2d};

static BAS_FCTS  lagrange3_2d = {"lagrange3_2d", 2, N_BAS3_2D, 3,
				 {1, 1, 2, 0}, /* VERTEX, CENTER, EDGE, FACE */
				 nil,
				 phi3_2d, grd_phi3_2d, D2_phi3_2d,
				 get_dof_indices3_2d, 
				 get_bound3_2d,
				 interpol3_2d,
				 interpol_d3_2d,
				 get_int_vec3_2d,
				 get_real_vec3_2d,
				 get_real_d_vec3_2d,
				 get_uchar_vec3_2d,
				 get_schar_vec3_2d,
				 real_refine_inter3_2d,
				 real_coarse_inter3_2d,
				 real_coarse_restr3_2d,
				 real_d_refine_inter3_2d,
				 real_d_coarse_inter3_2d,
				 real_d_coarse_restr3_2d,
				 bary3_2d,};

