////////////////////////////////////////////////////////////
//
// SFML - Simple and Fast Multimedia Library
// Copyright (C) 2007-2009 Lucas Soltic (ceylow@gmail.com) and Laurent Gomila (laurent.gom@gmail.com)
//
// This software is provided 'as-is', without any express or implied warranty.
// In no event will the authors be held liable for any damages arising from the use of this software.
//
// Permission is granted to anyone to use this software for any purpose,
// including commercial applications, and to alter it and redistribute it freely,
// subject to the following restrictions:
//
// 1. The origin of this software must not be misrepresented;
//    you must not claim that you wrote the original software.
//    If you use this software in a product, an acknowledgment
//    in the product documentation would be appreciated but is not required.
//
// 2. Altered source versions must be plainly marked as such,
//    and must not be misrepresented as being the original software.
//
// 3. This notice may not be removed or altered from any source distribution.
//
////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////
// Headers
////////////////////////////////////////////////////////////
#import <Cocoa/Cocoa.h>
#import <SFML/Window/Cocoa/WindowImplCocoa.hpp>


////////////////////////////////////////////////////////////
/// Window independant OpenGL context class
////////////////////////////////////////////////////////////
@interface sfPrivGLContext : NSOpenGLContext

////////////////////////////////////////////////////////////
/// Return the shared OpenGL context instance (making one if needed)
////////////////////////////////////////////////////////////
+ (id)sharedContext;

////////////////////////////////////////////////////////////
/// Make a new OpenGL context according to the @attribs settings
/// and the shared context @context
////////////////////////////////////////////////////////////
- (id)initWithAttributes:(sf::WindowSettings&)attribs
					mode:(const sf::VideoMode&)mode
		   sharedContext:(sfPrivGLContext *)context;

@end


////////////////////////////////////////////////////////////
/// Customized Cocoa OpenGL view
////////////////////////////////////////////////////////////
@interface sfPrivGLView : NSOpenGLView
{
	sf::priv::WindowImplCocoa *myDelegate;
	sfPrivGLContext *myGLContext;
}

////////////////////////////////////////////////////////////
/// Make a new view according the the rect @frame,
/// the video mode @mode, the window settings @settings
/// @delegate must not be null
////////////////////////////////////////////////////////////
- (id)initWithFrame:(NSRect)frame
			   mode:(const sf::VideoMode&)mode
		   settings:(sf::WindowSettings&)settings;


////////////////////////////////////////////////////////////
/// Sets @aDelegate as the view delegate
////////////////////////////////////////////////////////////
- (void)setDelegate:(sf::priv::WindowImplCocoa *)aDelegate;

////////////////////////////////////////////////////////////
/// Returns the view delegate
////////////////////////////////////////////////////////////
- (sf::priv::WindowImplCocoa *)delegate;

////////////////////////////////////////////////////////////
/// Finish view setting (after having added it to the window)
////////////////////////////////////////////////////////////
- (void)finishInitialization;

////////////////////////////////////////////////////////////
/// Forward call to en/disable vertical synchronization
////////////////////////////////////////////////////////////
- (void)enableVerticalSync:(bool)flag;

////////////////////////////////////////////////////////////
/// Forward call to set the OpenGL context as active according to @flag
////////////////////////////////////////////////////////////
- (void)setActive:(bool)flag;

////////////////////////////////////////////////////////////
/// Forward call to flush the OpenGL context
////////////////////////////////////////////////////////////
- (void)flushBuffer;

@end

////////////////////////////////////////////////////////////
/// Parent class for handling general SFML window stuff
////////////////////////////////////////////////////////////
@interface sfPrivWindow : NSObject
{
@private
	NSWindow *myWindow;
	sfPrivGLView *myView;
}

////////////////////////////////////////////////////////////
/// Return a reference to the internal Cocoa window
////////////////////////////////////////////////////////////
- (NSWindow *)window;

////////////////////////////////////////////////////////////
/// Return a reference to the internal Cocoa OpenGL view
////////////////////////////////////////////////////////////
- (sfPrivGLView *)view;

////////////////////////////////////////////////////////////
/// Sets @aDelegate as the window delegate
////////////////////////////////////////////////////////////
- (void)setDelegate:(sf::priv::WindowImplCocoa *)aDelegate;

////////////////////////////////////////////////////////////
/// Returns the window delegate
////////////////////////////////////////////////////////////
- (sf::priv::WindowImplCocoa *)delegate;

////////////////////////////////////////////////////////////
/// Forward call to set the window position on screen
////////////////////////////////////////////////////////////
- (void)setPosition:(NSPoint)pos;

////////////////////////////////////////////////////////////
/// Forward call to set the window size
////////////////////////////////////////////////////////////
- (void)setSize:(NSSize)size;

////////////////////////////////////////////////////////////
/// Return the mouse location relative to the internal window
////////////////////////////////////////////////////////////
- (NSPoint)mouseLocation;

////////////////////////////////////////////////////////////
/// Return whether the mouse is on our window
////////////////////////////////////////////////////////////
- (BOOL)mouseInside;

////////////////////////////////////////////////////////////
/// Close or open the window
////////////////////////////////////////////////////////////
- (void)show:(bool)flag;

////////////////////////////////////////////////////////////
/// Forward call to en/disable the OpenGL view vertical synchronization
////////////////////////////////////////////////////////////
- (void)enableVerticalSync:(bool)flag;

////////////////////////////////////////////////////////////
/// Forward 'setActive' call the the OpenGL view
////////////////////////////////////////////////////////////
- (void)setActive:(bool)flag;

////////////////////////////////////////////////////////////
/// Forward call to flush the OpenGL view
////////////////////////////////////////////////////////////
- (void)flushBuffer;

@end

////////////////////////////////////////////////////////////
/// Class for creating new SFML windows from informations
////////////////////////////////////////////////////////////
@interface sfPrivOwnedWindow : sfPrivWindow
{
@private
	sf::VideoMode myFullscreenMode;
	bool myIsFullscreen;
}

////////////////////////////////////////////////////////////
/// Creates and returns a new SFML window handler with
/// the given parameters
////////////////////////////////////////////////////////////
- (id)initWithVideoMode:(sf::VideoMode&)aMode
			   settings:(sf::WindowSettings&)someSettings
				  style:(unsigned long)aStyle
				  title:(NSString *)aTitle;

////////////////////////////////////////////////////////////
/// Returns the window's fullscreen state
////////////////////////////////////////////////////////////
- (BOOL)isFullscreen;

@end


////////////////////////////////////////////////////////////
/// Class for creating SFML windows from Cocoa windows
////////////////////////////////////////////////////////////
@interface sfPrivImportedWindow : sfPrivWindow

////////////////////////////////////////////////////////////
/// Returns a new SFML window handler with the given window
/// and parameters
////////////////////////////////////////////////////////////
- (id)initWithWindow:(NSWindow *)aWindow
			settings:(sf::WindowSettings&)someSettings;

@end


////////////////////////////////////////////////////////////
/// Class for creating SFML windows from Cocoa views
////////////////////////////////////////////////////////////
@interface sfPrivImportedView : sfPrivWindow
{
	NSView *parentView;
}

////////////////////////////////////////////////////////////
/// Returns a new SFML window handler with the given view
/// and parameters
////////////////////////////////////////////////////////////
- (id)initWithView:(NSView *)aView
		  settings:(sf::WindowSettings&)someSettings;

@end
