#    Copyright 2015 SimpliVity Corp.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
#    under the License.

from oslo_versionedobjects import fields

from cinder import db
from cinder import exception
from cinder.i18n import _
from cinder import objects
from cinder.objects import base
from cinder.objects import fields as c_fields


@base.CinderObjectRegistry.register
class VolumeAttachment(base.CinderPersistentObject, base.CinderObject,
                       base.CinderObjectDictCompat,
                       base.CinderComparableObject):
    # Version 1.0: Initial version
    VERSION = '1.0'

    fields = {
        'id': fields.UUIDField(),
        'volume_id': fields.UUIDField(),
        'instance_uuid': fields.UUIDField(nullable=True),
        'attached_host': fields.StringField(nullable=True),
        'mountpoint': fields.StringField(nullable=True),

        'attach_time': fields.DateTimeField(nullable=True),
        'detach_time': fields.DateTimeField(nullable=True),

        'attach_status': c_fields.VolumeAttachStatusField(nullable=True),
        'attach_mode': fields.StringField(nullable=True),
    }

    @staticmethod
    def _from_db_object(context, attachment, db_attachment):
        for name, field in attachment.fields.items():
            value = db_attachment.get(name)
            if isinstance(field, fields.IntegerField):
                value = value or 0
            attachment[name] = value

        attachment._context = context
        attachment.obj_reset_changes()
        return attachment

    def save(self):
        updates = self.cinder_obj_get_changes()
        if updates:
            db.volume_attachment_update(self._context, self.id, updates)
            self.obj_reset_changes()

    def finish_attach(self, instance_uuid, host_name,
                      mount_point, attach_mode='rw'):
        with self.obj_as_admin():
            db_volume, updated_values = db.volume_attached(
                self._context, self.id,
                instance_uuid, host_name,
                mount_point, attach_mode)
        self.update(updated_values)
        self.obj_reset_changes(updated_values.keys())
        return objects.Volume._from_db_object(self._context,
                                              objects.Volume(),
                                              db_volume)

    def create(self):
        if self.obj_attr_is_set('id'):
            raise exception.ObjectActionError(action='create',
                                              reason=_('already created'))
        updates = self.cinder_obj_get_changes()
        with self.obj_as_admin():
            db_attachment = db.volume_attach(self._context, updates)
        self._from_db_object(self._context, self, db_attachment)


@base.CinderObjectRegistry.register
class VolumeAttachmentList(base.ObjectListBase, base.CinderObject):
    # Versoin 1.0: Iniitial version
    # Version 1.1: Remove volume_id in get_by_host|instance
    VERSION = '1.1'

    fields = {
        'objects': fields.ListOfObjectsField('VolumeAttachment'),
    }

    @classmethod
    def get_all_by_volume_id(cls, context, volume_id):
        attachments = db.volume_attachment_get_all_by_volume_id(context,
                                                                volume_id)
        return base.obj_make_list(context,
                                  cls(context),
                                  objects.VolumeAttachment,
                                  attachments)

    @classmethod
    def get_all_by_host(cls, context, host):
        attachments = db.volume_attachment_get_all_by_host(context,
                                                           host)
        return base.obj_make_list(context, cls(context),
                                  objects.VolumeAttachment, attachments)

    @classmethod
    def get_all_by_instance_uuid(cls, context, instance_uuid):
        attachments = db.volume_attachment_get_all_by_instance_uuid(
            context, instance_uuid)
        return base.obj_make_list(context, cls(context),
                                  objects.VolumeAttachment, attachments)
