/******************************** LICENSE ********************************

 Copyright 2007 European Centre for Medium-Range Weather Forecasts (ECMWF)

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at 

    http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.

 ******************************** LICENSE ********************************/

/*! \file VisualAction.h
    \brief Implementation of the Template class ActionNode.
    
    Magics Team - ECMWF 2007
    
    Started: Tue 6-Mar-2007
    
    Changes:
    
*/ 

#include "VisualAction.h"
#include "LegendVisitor.h"
#include "Timer.h"
#include "TextVisitor.h"
#include "SceneVisitor.h"
#include "AnimationRules.h"
#include "Layer.h"
#include "MagnifierVisitor.h"
#include "HistoVisitor.h"

using namespace magics;


VisualAction::VisualAction() : data_(0)
{
}


VisualAction::~VisualAction()
{
	MagLog::debug() << "delete VisualAction" << endl;
	if (data_) delete data_;
}

/*!
 Class information are given to the output-stream.
*/	

void VisualAction::print(ostream& out)  const
{
	out << "ActionNode";
}


void VisualAction::getReady()
{
	MagLog::dev() << "ActionNode::getReady()\n";
}


void VisualAction::release()
{

	if ( data_ )
		data_->release();
}

void VisualAction::visit(DateDescription& timestamp) {
	if ( data_ )
		timestamp = data_->timeStamp();
}


void VisualAction::visit(LevelDescription& level) {
	if ( data_ )
		level = data_->level();
}




void VisualAction::visit(DrawingVisitor& drawing)
{
	if ( !data_ || visdefs_.empty() )
	{
		MagLog::warning() << " No proper action defined!" << endl;
		return;
	}
	data_->getReady(drawing.transformation());

	for (  vector<Visdef* >::iterator visdef = visdefs_.begin(); visdef != visdefs_.end(); ++visdef)
	{
		Timer timer("plotting", "time spent in plotting");
		(**visdef)(*data_, drawing.layout()); // Now the visualObject ahs the responsability to reproject!
	}
}


void VisualAction::visit(HistoVisitor& drawing)
{
	if ( !visdefs_.empty() ) {
		// We only send it to the first action...


		 
		drawing.basic(true);
		drawing.dataLayoutTransformation(&transformation());
		 
		for(  vector<Visdef* >::iterator visdef = visdefs_.begin(); visdef != visdefs_.end(); ++visdef)
		{
			if((*visdef)->iconName() == drawing.dataVisdefIcon().iconName() &&
			   (*visdef)->iconClass() == drawing.dataVisdefIcon().iconClass() && 
			   ! (*visdef)->iconClass().empty())
			{
				drawing.basic(false);
				(*visdef)->visit(*data_, drawing);
				return;	
			}  
			  
			  	
		}


		visdefs_.front()->visit(*data_, drawing);
	}
}



void VisualAction::visit(Transformation& transformation)
{
	if ( !data_ || visdefs_.empty() ) {
		MagLog::warning() << " No proper action defined!" << endl;
		return;
	}
	data_->visit(transformation);	
	for ( vector<Visdef* >::iterator visdef = visdefs_.begin(); visdef != visdefs_.end(); ++visdef)
		(**visdef).visit(transformation, *data_);
}
 
 

void VisualAction::visit(LegendVisitor& legend)
{
	unsigned int entries = legend.size();
	bool needEmpty = false;
	legend.transformation(&transformation());
	for ( vector<Visdef* >::iterator visdef = visdefs_.begin(); visdef != visdefs_.end(); ++visdef) {
		if ( needEmpty ) 
			legend.add(new EmptyEntry() );
		(**visdef).visit(*data_, legend);
		needEmpty = ( entries != legend.size() );
		entries = legend.size();
	}
	data_->visit(legend);
}


void VisualAction::visit(TextVisitor& title)
{
	if ( data_ ) data_->visit(title);
}

void VisualAction::visit(TopAxisVisitor& top)
{
	for ( vector<Visdef* >::iterator visdef = this->visdefs_.begin(); visdef != this->visdefs_.end(); ++visdef) {
			(*visdef)->visit(top);
	}
}

void VisualAction::visit(MetaDataVisitor& infos)
{
	if ( data_ ) data_->visit(infos);
	for ( vector<Visdef* >::iterator visdef = this->visdefs_.begin(); visdef != this->visdefs_.end(); ++visdef) {
		(*visdef)->visit(infos);
	}
}


void VisualAction::visit(MetaDataCollector& infos)
{
	infos.transformation(&transformation());
	if ( data_ ) data_->visit(infos);
}


void VisualAction::visit(ValuesCollector& infos)
{
	infos.transformation(&transformation());
	if ( data_ ) data_->visit(infos);
}

void VisualAction::visit(DataIndexCollector& infos)
{
	if ( data_ ) data_->visit(infos);
}

void VisualAction::visit(MagnifierCollector& infos)
{
	infos.transformation(&transformation());
	if ( data_ ) data_->visit(infos);
}


void VisualAction::visit(AnimationRules& rules)
{
	if ( data_ ) data_->visit(rules);
}

static string unknown("unknown");
const string& VisualAction::name()
{

	return ( data_ ) ? data_->name() : unknown;
}


const string& VisualAction::id() {
	static string name("unknown");
	return ( data_ ) ? data_->layerId() : unknown;
}


	

VisualAnimation::VisualAnimation():loop_(0)
{
}


VisualAnimation::~VisualAnimation()
{
	cout << "delete VisualAnimation" << endl;
	if (loop_) delete loop_;
}


void VisualAnimation::prepare()
{
	if ( !this->empty() )
		return;
	
	layer_ = new StepLayer();
	layer_->name(loop_->name());
	layer_->id(loop_->name()); 
	layer_->uniqueId(loop_->iconId());
	loop_->visit(*layer_);
	
	for ( vector<Visdef* >::iterator visdef = this->visdefs_.begin(); visdef != this->visdefs_.end(); ++visdef)
	{    		
		(*visdef)->visit(*layer_);
	}
				
	loop_->setToFirst(); 
	while ( loop_->hasMore() )
	{
		MagLog::dev() << "New Frame" << endl; 

		VisualAction* action = new VisualAction();
		action->parent(this);
		action->data(loop_->current());

		for ( vector<Visdef* >::iterator visdef = this->visdefs_.begin(); visdef != this->visdefs_.end(); ++visdef) {
			action->visdef(*visdef);
		}

		layer_->addStep(action);



		loop_->next();
	}
}



void VisualAnimation::visit(MetaDataVisitor&)
{
}


void VisualAnimation::visit(AnimationRules& rules)
{
	prepare();
	for ( iterator entry = this->begin(); entry != this->end(); ++entry)
		(*entry)->visit(rules);
	rules.add(*this->layer_);
}



void VisualAnimation::visit(Transformation& transformation)
{
	assert(loop_);
	loop_->visit(transformation);
} 


void VisualAnimation::visit(SceneLayer& layer, vector<LayoutVisitor*>& visitors)
{
   	layer.add(layer_);
 	for  (vector<LayoutVisitor*>::iterator visitor = visitors.begin(); visitor != visitors.end(); ++visitor) {
   		layer.addVisitor(*visitor); 
   		layer_->addVisitor(*visitor);  
   	}
}

bool VisualAction::needLegend()
{
	for(vector<Visdef* >::iterator visdef = this->visdefs_.begin(); visdef != this->visdefs_.end(); ++visdef)
		{
			if ( (*visdef)->needLegend() )
				return true;
		}
	return false;
}

void VisualAction::visit(SceneLayer& layer, vector<LayoutVisitor*>& visitors)
{
	StaticLayer* plot = new StaticLayer(this);
	

	if ( data_ ) 
		data_->visit(*plot);
 	layer.add(plot);

	for  (vector<LayoutVisitor*>::iterator visitor = visitors.begin(); visitor != visitors.end(); ++visitor) {
		plot->set(*visitor);
		(*visitor)->visit(*this);
	}

	for(vector<Visdef* >::iterator visdef = this->visdefs_.begin(); visdef != this->visdefs_.end(); ++visdef)
	{    
		(*visdef)->visit(*plot);
	}


}

