/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "AudioDialog.h"
#include "GlobalIcon.h"
#include "AudioBase.h"
#include "AudioDrivers.h"

#include <qcombobox.h>
#include <qradiobutton.h>

static int frequencyList[] = 
{
    // in Hz
    8000, 11025, 16500, 22050, 27500, 32000, 37500, 44100, 48000,
    0  // end
};

static int bufferSizeList[] = 
{
    // in Bytes
    256, 512, 1024, 2048, 4096, 8192, 16384, 32768,
    0  // end
};

AudioDialog::AudioDialog(QWidget* parent, const char* name) 
: AudioDialogData(parent,name)
{
    setIcon(*mainIcon);
    // Fill combo box with audio driver names.
    AudioDrivers::List::iterator it;
    for ( it = AudioDrivers::begin(); it != AudioDrivers::end(); it++ )
    {
        driverBox->insertItem( it->name );
    }
    setButtons();
}

void AudioDialog::setButtons()
{
    bit8Btn->setChecked(config.precision==AudioConfig::BITS_8);
    bit16Btn->setChecked(config.precision==AudioConfig::BITS_16);
    monoBtn->setChecked(config.channels==AudioConfig::MONO);
    stereoBtn->setChecked(config.channels==AudioConfig::STEREO);
    unsignedBtn->setChecked(config.encoding==AudioConfig::UNSIGNED_PCM);
    signedBtn->setChecked(config.encoding==AudioConfig::SIGNED_PCM);
    
    int i = 0, foundIndex = 0;
    // Activate startup value in frequency box.
    while (frequencyList[i] != 0)
    {
        if (config.frequency >= frequencyList[i])
        {
            foundIndex = i;
        }
        i++;
    };
    frequencyBox->setCurrentItem(foundIndex);

    i = 0, foundIndex = 0;
    // Activate startup value in buffer size box.
    while (bufferSizeList[i] != 0)
    {
        if (config.bufSize >= bufferSizeList[i])
        {
            foundIndex = i;
        }
        i++;
    };
    bufferSizeBox->setCurrentItem(foundIndex);

    i = 0;
    // Activate startup value in driver box.
    while ( i < driverBox->count() )
    {
        if ( driverBox->text(i) == AudioDrivers::getName() )
        {
            driverBox->setCurrentItem(i);
            break;
        }
        ++i;
    }
}

void AudioDialog::setConfig(const AudioConfig& inConfig)
{
    // Copy without verification.
    config = inConfig;
    setButtons();
}

const AudioConfig& AudioDialog::getConfig() const
{
    return config;
}

void AudioDialog::setPrecision(int)
{
    if (bit8Btn->isChecked())
        config.precision = AudioConfig::BITS_8;
    else  // if (bit16Btn->isChecked())
        config.precision = AudioConfig::BITS_16;
}

void AudioDialog::setEncoding(int)
{
    if (unsignedBtn->isChecked())
      config.encoding = AudioConfig::UNSIGNED_PCM;
    else
      config.encoding = AudioConfig::SIGNED_PCM;
}

void AudioDialog::setChannels(int)
{
    if (monoBtn->isChecked())
        config.channels = AudioConfig::MONO;
    else  // if (stereoBtn->isChecked())
        config.channels = AudioConfig::STEREO;
}

void AudioDialog::setFrequency(int)
{
    config.frequency = frequencyList[frequencyBox->currentItem()];
}

void AudioDialog::setBufferSize(int)
{
    config.bufSize = bufferSizeList[bufferSizeBox->currentItem()];
}

void AudioDialog::setDriver( const QString& name )
{
    AudioDrivers::setDriver( name.data() );
}
